/***********************************************************
 *
 * d_UART_HAL.c
 * SYSTART GmbH
 * MIGA, 09.11.2021
 *
 ***********************************************************/

/***********************************************************
 *
 * @description: This file implements an UART ring buffer via
 * an Interrupt. uart_init must be called to initialize the
 * rx-buffers and activate the hal-receive interrupt.
 * The HAL_UART_RxCpltCallback() function is called when 1 byte
 * is successfully received, the function increases the wr_ptr
 * and calls the hal-receive interrupt function again.
 * Retrieving the items from the ring buffer is available byte-,
 * string- and array-wise.
 *
 ***********************************************************/

 /***********************************************************
 * Includes
 ***********************************************************/
#include "driver/d_UART_HAL.h"

extern UART_HandleTypeDef huart1;
extern UART_HandleTypeDef huart3;
extern UART_HandleTypeDef huart6;
 /***********************************************************
 * Function Definition
 ***********************************************************/
void uart_init(void)
{
	UART_NAME_t nr;
  for(nr=0;nr<UART_ANZ;nr++) {

  	/*
  	 * HAL takes over initialization and interrupt handler
  	 */

    //Prepare RX-Buffer
    UART_RX[nr].rx_buffer[0]=RX_END_CHR;
    UART_RX[nr].wr_ptr=0;
    UART_RX[nr].rd_ptr=0;
    UART_RX[nr].status=RX_EMPTY;

  }
  HAL_UART_Receive_IT(&huart1, &UART_RX[COM1].rx_buffer[UART_RX[COM1].wr_ptr], 1);
  HAL_UART_Receive_IT(&huart3, &UART_RX[COM3].rx_buffer[UART_RX[COM3].wr_ptr], 1);
  HAL_UART_Receive_IT(&huart6, &UART_RX[COM6].rx_buffer[UART_RX[COM6].wr_ptr], 1);
}

void uart_SendByte(UART_NAME_t uart, uint8_t byte){
	UART_HandleTypeDef huart;
	switch (uart) {
		case COM1:
			huart = huart1;
			break;
		case COM3:
			huart = huart3;
			break;
		case COM6:
			huart = huart6;
			break;
		default:
			return;
			break;
	}

	HAL_UART_Transmit(&huart, &byte, 1, UART_TIMEOUT);
}

void uart_SendString(UART_NAME_t uart, uint8_t *ptr, UART_LASTBYTE_t end_cmd)
{
	UART_HandleTypeDef huart;
	uint8_t endCmd_len = 0;
	uint8_t endCmd[2];
	switch (uart) {
		case COM1:
			huart = huart1;
			break;
		case COM3:
			huart = huart3;
			break;
		case COM6:
			huart = huart6;
			break;
		default:
			return;
			break;
	}
  // send whole string
  while (*ptr != 0) {
    HAL_UART_Transmit(&huart, ptr, 1, UART_TIMEOUT);
    ptr++;
  }
  // if wanted, send end command
  if(end_cmd==LFCR) {
  	endCmd[0] = 0x0A; // send LineFeed
  	endCmd[1] = 0x0D; // send CariageReturn
  	endCmd_len = 2;
  }
  else if(end_cmd==CRLF) {
  	endCmd[0] = 0x0D; // send CariageReturn
  	endCmd[1] = 0x0A; // send LineFeed
  	endCmd_len = 2;
  }
  else if(end_cmd==LF) {
  	endCmd[0] = 0x0A; // send LineFeed
  	endCmd_len = 1;
  }
  else if(end_cmd==CR) {
  	endCmd[0] = 0x0D; // send CariageReturn
  	endCmd_len = 1;
  }
  HAL_UART_Transmit(&huart, endCmd, endCmd_len, UART_TIMEOUT);
}

UART_RXSTATUS_t uart_ReceiveString(UART_NAME_t uart, char *ptr)
{
  UART_RXSTATUS_t ret_wert=RX_EMPTY;
  uint16_t rd_pos,wr_pos,mem_rd_pos,n;
  uint8_t wert,ok,vorletzterwert;

  // read current pointer
  rd_pos=UART_RX[uart].rd_ptr;
  wr_pos=UART_RX[uart].wr_ptr;
  // check if there is data in buffer
  ok=0;
  if(rd_pos!=wr_pos) {
    // search for end
    while(rd_pos!=wr_pos) {
      wert=UART_RX[uart].rx_buffer[rd_pos];
      if(rd_pos){
				vorletzterwert = UART_RX[uart].rx_buffer[rd_pos-1];
			}
      else{
      	vorletzterwert = UART_RX[uart].rx_buffer[RX_BUF_SIZE-1];
      }
      if(wert==RX_END_CHR) {
      	mem_rd_pos = -1;
        // end detected
        ok=1;
        break;
      }
      if(vorletzterwert==0x5c && wert == 0x6e){
      	// end detected
      	// save last relevant char
      	mem_rd_pos = rd_pos - 1;
				ok=1;
				break;
      }
      // increase read pointer
      rd_pos++;
      if(rd_pos>=RX_BUF_SIZE) rd_pos=0;
    }

    // if end detected, read string
    if(ok==1) {
      // end detected
      ret_wert=RX_READY;
      // read whole string
      rd_pos=UART_RX[uart].rd_ptr;
      n=0;
      while(rd_pos!=wr_pos) {
        wert=UART_RX[uart].rx_buffer[rd_pos];
        // increase read pointer
        rd_pos++;
        if(rd_pos>=(RX_BUF_SIZE-1)) rd_pos=0;
        // store character
        ptr[n]=wert;
				n++;
      	if(rd_pos == mem_rd_pos) {
      		// update read pointer
      		UART_RX[uart].rd_ptr = rd_pos + 2;
      		break;
				}
        if(wert==RX_END_CHR) {
        	// update read pointer
        	UART_RX[uart].rd_ptr=rd_pos;
          break;
        }
      }
      // mark end of string
      ptr[n]=0x00;
    }
    else {
      // no end detected
      ret_wert=RX_EMPTY;
    }
  }
  else {
    // check for overflow
    if(UART_RX[uart].status==RX_FULL) {
      ret_wert=RX_FULL;
      UART_RX[uart].status=RX_EMPTY;
    }
  }

  return(ret_wert);
}

void uart_SendArray(UART_NAME_t uart, uint8_t *data, uint16_t cnt, Endianness_t endianness)
{
  if(cnt==0) return;

  uint32_t n;
	UART_HandleTypeDef huart;
	switch (uart) {
		case COM1:
			huart = huart1;
			break;
		case COM3:
			huart = huart3;
			break;
		case COM6:
			huart = huart6;
			break;
		default:
			return;
			break;
	}

  // send all data
  if(LITTLE_ENDIAN_T == endianness){
	  n = cnt-1;
	  do{
		  HAL_UART_Transmit(&huart, &data[n], 1, UART_TIMEOUT);
	  } while(n--);
  }
  else if(BIG_ENDIAN_T == endianness){
	  for(n=0;n<cnt;n++) {
	  	HAL_UART_Transmit(&huart, &data[n], 1, UART_TIMEOUT);
	  }
  }
}

uint32_t uart_ReceiveArray(UART_NAME_t uart, uint8_t *data)
{
  uint32_t n=0;
  uint16_t rd_pos,wr_pos;
  uint8_t wert;

  // read current pointer
  rd_pos=UART_RX[uart].rd_ptr;
  wr_pos=UART_RX[uart].wr_ptr;
  // check if there is data in buffer
  n=0;
  if(rd_pos!=wr_pos) {
    // read all data
    while(rd_pos!=wr_pos) {
      wert=UART_RX[uart].rx_buffer[rd_pos];
      // increase read pointer
      rd_pos++;
      if(rd_pos>=RX_BUF_SIZE) rd_pos=0;
      // store data
      data[n]=wert;
      n++;
    }
    // update read pointer
    UART_RX[uart].rd_ptr=rd_pos;
  }

  return(n);
}

uint8_t uart_Receive2Byte(UART_NAME_t uart, uint16_t *data, Endianness_t endianness)
{
	uint8_t count = 0;
	uint16_t rd_pos, wr_pos;
	uint8_t wert1, wert2;

	// read current pointer
	rd_pos=UART_RX[uart].rd_ptr;
	wr_pos=UART_RX[uart].wr_ptr;
	// check if there is data in buffer
	if(rd_pos!=wr_pos) {
		//read data
		wert1 = UART_RX[uart].rx_buffer[rd_pos];
		// increase read pointer
		rd_pos++;
		count++;
		if(rd_pos>=RX_BUF_SIZE) rd_pos=0;

		wert2 = UART_RX[uart].rx_buffer[rd_pos];
		rd_pos++;
		count++;
		if(rd_pos>=RX_BUF_SIZE) rd_pos=0;
		if(endianness == BIG_ENDIAN_T) *data = wert1 | wert2 << 8;
		else if(endianness == LITTLE_ENDIAN_T) *data = wert1 << 8 | wert2;
	}
  	if(rd_pos > wr_pos) return 0;
    // update read pointer
    UART_RX[uart].rd_ptr=rd_pos;
    return count;
}

uint8_t uart_Receive4Byte(UART_NAME_t uart, uint8_t *data, Endianness_t endianness)
{
	uint8_t count = 0;
	uint16_t rd_pos, wr_pos;
	uint8_t wert1, wert2, wert3, wert4;

	// read current pointer
	rd_pos=UART_RX[uart].rd_ptr;
	wr_pos=UART_RX[uart].wr_ptr;
	// check if there is data in buffer
	if(rd_pos!=wr_pos) {
	// read data
	wert1 = UART_RX[uart].rx_buffer[rd_pos];
	// increase read pointer
	rd_pos++;
	count++;
	if(rd_pos>=RX_BUF_SIZE) rd_pos=0;

	wert2 = UART_RX[uart].rx_buffer[rd_pos];
	rd_pos++;
	count++;
	if(rd_pos>=RX_BUF_SIZE) rd_pos=0;

	wert3 = UART_RX[uart].rx_buffer[rd_pos];
	rd_pos++;
	count++;
	if(rd_pos>=RX_BUF_SIZE) rd_pos=0;

	wert4 = UART_RX[uart].rx_buffer[rd_pos];
	rd_pos++;
	count++;
	if(rd_pos>=RX_BUF_SIZE) rd_pos=0;

		if(endianness == BIG_ENDIAN_T){
			data[0] = wert1;
			data[1] = wert2;
			data[2] = wert3;
			data[3] = wert4;
		}
		else if(endianness == LITTLE_ENDIAN_T){
			data[3] = wert1;
			data[2] = wert2;
			data[1] = wert3;
			data[0] = wert4;
		}
	}
	if(rd_pos > wr_pos) return 0;
    // update read pointer
    UART_RX[uart].rd_ptr=rd_pos;
    return count;
}

uint8_t uart_Receive1Byte(UART_NAME_t uart, uint8_t *data)
{
	uint8_t count = 0;
  uint16_t rd_pos,wr_pos;

  // read current pointer
  rd_pos=UART_RX[uart].rd_ptr;
  wr_pos=UART_RX[uart].wr_ptr;

  // check if there is data in buffer
  if(rd_pos!=wr_pos) {
    // read data
	 *data = UART_RX[uart].rx_buffer[rd_pos];
	// increase read pointer
	rd_pos++;
	count++;
	if(rd_pos>=RX_BUF_SIZE) rd_pos=0;
  }
  // update read pointer
  UART_RX[uart].rd_ptr=rd_pos;
  return count;
}

uint16_t get_rd_ptr(UART_NAME_t uart){
	return UART_RX[uart].rd_ptr;
}

uint16_t get_wr_ptr(UART_NAME_t uart){
	return UART_RX[uart].wr_ptr;
}

void reset_ringbuffer(UART_NAME_t uart){
	UART_RX[uart].rd_ptr = UART_RX[uart].wr_ptr;
}

void HAL_UART_RxCpltCallback(UART_HandleTypeDef *huart)
{
	uint16_t wr_pos;
	UART_NAME_t nr;
	if(huart->Instance == USART1){
		nr = COM1;
	}
	if(huart->Instance == USART3){
		nr = COM3;
	}
	if(huart->Instance == USART6){
		nr = COM6;
	}

	wr_pos = UART_RX[nr].wr_ptr;

	wr_pos++;
	if(wr_pos>=RX_BUF_SIZE){
		wr_pos=0;
	}
	UART_RX[nr].wr_ptr=wr_pos;

	// check for overflow
	if(UART_RX[nr].wr_ptr==UART_RX[nr].rd_ptr) {
	UART_RX[nr].status=RX_FULL;
	}
	HAL_UART_Receive_IT(huart, &UART_RX[nr].rx_buffer[wr_pos], 1);
}
