/***********************************************************
 *
 * d_Touch_FT5316_ILI2130_HAL.c
 * SYSTART GmbH
 * MIGA, 24.10.2021
 *
 ***********************************************************/

/***********************************************************
 *
 * @description: This file implements the I2C communication
 * with the Display-touch-panel. There are two possible
 * Touch driver ICs on the display board. The FT5316 or the
 * ILI2130, the driver file can communicate with both of them.
 * It is possible to implement polling with this driver file.
 * By default the EXTI3 on Pin PD3 is used. The ISR calls the
 * function retrieve_touch_points() in this file and the
 * global structure Touch is filled with touch-points.
 *
 ***********************************************************/

/***********************************************************
 * Includes
 ***********************************************************/
#include <driver/d_Touch_FT5316_ILI2130_HAL.h>
#include "global_Display_Touch_HAL.h"

extern I2C_HandleTypeDef hi2c1;

/***********************************************************
 * Declaration internal functions
 ***********************************************************/
bool AreaCheck ( uint16_t LeftX, uint16_t LeftY, uint16_t RightX, uint16_t RightY );

/***********************************************************
 * Global variables
 ***********************************************************/
static TouchXY_t Touch = {0};

/***********************************************************
 * Function definitions
 ***********************************************************/
void retrieve_touch_points()
{
  uint8_t buffer[65];
  HAL_StatusTypeDef ret;
  float xTemp, yTemp;

  buffer[0] = 0x00;
  /* Try to communicate with FT5316 */
  ret = HAL_I2C_Master_Transmit(&hi2c1, 0x70, buffer, 1, HAL_MAX_DELAY);
  if (ret == HAL_ERROR) {
  	/* Try to communicate with ILI2130 */
    ret = HAL_I2C_Master_Receive(&hi2c1, 0x83, buffer, 64, HAL_MAX_DELAY);
    /* Both failed */
    if (ret != HAL_OK) {
      return;
    }
    /* ILI2130 detected, parse response */
    if ((buffer[1] >> 6) == 0)
      Touch.event = RELEASE;
    else if ((buffer[1] >> 6) == 1)
      Touch.event = TOUCH;
    else
      Touch.event = 0xFF;
    xTemp = buffer[3] << 8 | buffer[2];
    yTemp = buffer[5] << 8 | buffer[4];
#ifdef DISPLAY_3_5_INCH
    Touch.xPos = (uint16_t)((xTemp * 320 / 16384) + 0.5);
    Touch.yPos = (uint16_t)((yTemp * 240 / 16384) + 0.5);
#endif
#ifdef DISPLAY_5_INCH
    Touch.xPos = (uint16_t)((xTemp * 800 / 16384) + 0.5);
    Touch.yPos = (uint16_t)((yTemp * 480 / 16384) + 0.5);
#endif
    return;
  }

  /* FT5316 detected, get Touch points */
  ret = HAL_I2C_Master_Receive(&hi2c1, 0x70, buffer, 30, HAL_MAX_DELAY);
  if (ret != HAL_OK) {
    return;
  }
  /* Parse response */
  Touch.event = (buffer[3] >> 6);

  xTemp = (buffer[3] & 0x0F) << 8 | buffer[4];
  yTemp = (buffer[5] & 0x0F) << 8 | buffer[6];
#ifdef DISPLAY_3_5_INCH
  Touch.xPos = (uint16_t)((xTemp * 25 / 70) + 0.5);
  Touch.yPos = (uint16_t)((yTemp * 15 / 40) + 0.5);
#endif
#ifdef DISPLAY_4_3_INCH
  Touch.xPos = (uint16_t)((xTemp * 3 / 8) + 0.5);
  Touch.yPos = (uint16_t)((yTemp * 17 / 48) + 0.5);
#endif
#ifdef DISPLAY_5_INCH
  Touch.xPos = (uint16_t)((xTemp * 25 / 56) + 0.5);
  Touch.yPos = (uint16_t)((yTemp * 15 / 32) + 0.5);
#endif
}

TouchXY_t* return_touch_struct(){
  return &Touch;
}

bool AreaCheck ( uint16_t LeftX, uint16_t TopY, uint16_t RightX, uint16_t BottomY )
{
  if ( LeftX < Touch.xPos &&
     TopY < Touch.yPos &&
     RightX > Touch.xPos &&
     BottomY > Touch.yPos )
  {
    return true;
  }
  return false;
}
