/***********************************************************
 *
 * d_Display_DMA2D_HAL.h
 * SYSTART GmbH
 * MIGA, 16.10.2021
 *
 ***********************************************************/

#ifndef __D_DISPLAY_DMA2D_HAL_H_
#define __D_DISPLAY_DMA2D_HAL_H_

/***********************************************************
 * Includes
 **********************************************************/
#include <stdbool.h>
#include "main.h"
#include "global_Display_Touch_HAL.h"

/***********************************************************
 * Function declarations
************************************************************/

/**
* @brief Logo Display Sequence
*
* @return The active display buffer currently in use
*
*/
Buffer_e Display_Startup_Sequence();

/**
* @brief Draw a pixel with color at (xpos|ypos) on layer/buffer
*
* @param xpos			x-position
* @param ypos			y-position
* @param color 		Color
* @param layer		Layer number (Layer_1, Layer_2)
* @param buffer		Buffer number (Buffer_1, Buffer_2)
*
*/
void DMA2D_DrawPixel (uint16_t xpos, uint16_t ypos, uint32_t color, Layer_e layer, Buffer_e buffer);

/**
* @brief Fill whole buffer with a color
*
* @param color 		Color
* @param layer		Layer number (Layer_1, Layer_2)
* @param buffer		Buffer number (Buffer_1, Buffer_2)
* @param waitForVsync		If true, wait for vertical sync in ltdc->CDSR register
*
*/
void DMA2D_Fill_Color(uint32_t color, Layer_e layer, Buffer_e buffer, bool waitForVsync);

/**
* @brief Draw a filled rectangle
*
* @param color		Color of rectangle
* @param xpos 		x-position
* @param ypos 		y-position
* @param width 		Length of rectangle in pixels
* @param height		Height of rectangle in pixels
* @param layer		Layer number (Layer_1, Layer_2)
* @param buffer		Buffer number (Buffer_1, Buffer_2)
* @param waitForVsync		If true, wait for vertical sync in ltdc->CDSR register
*
*/
void DMA2D_Draw_FilledRectangle(uint32_t color, uint16_t xpos, uint16_t ypos, uint16_t width, uint16_t height, Layer_e layer, Buffer_e buffer, bool waitForVsync);

/**
* @brief Fill part of buffer with an image
*
* @param xpos 					x-position
* @param ypos 					y-position
* @param xsize 					Width of image
* @param ysize					Height of image
* @param alpha					Alpha value
* @param alpha_mode			Alpha mode
* @param addr_image 		Address of the image
* @param source_format 	Color format of the image (CM_RGB565, CM_RGB888, ...)
* @param layer					Layer number (Layer_1, Layer_2)
* @param buffer					Buffer number (Buffer_1, Buffer_2)
* @param waitForVsync		If true, wait for vertical sync in ltdc->CDSR register
*
*/
void DMA2D_Draw_Image(uint16_t xpos, uint16_t ypos, uint16_t xsize, uint16_t ysize, uint32_t alpha, uint32_t alpha_mode, uint32_t addr_image, uint32_t source_format, Layer_e layer, Buffer_e buffer, bool waitForVsync);

/**
* @brief Draw horizontal line
*
* @param xpos 		x-position
* @param ypos 		y-position
* @param length 	Length of line
* @param width		Width of line
* @param color		Color of line
* @param layer		Layer number (Layer_1, Layer_2)
* @param buffer		Buffer number (Buffer_1, Buffer_2)
*
*/
void DMA2D_Draw_X_Line(uint16_t xpos, uint16_t ypos, uint16_t length, uint16_t width, uint32_t color, Layer_e layer, Buffer_e buffer);

/**
* @brief Draw vertical line
*
* @param xpos 		x-position
* @param ypos 		y-position
* @param length 	Length of line
* @param width		Width of line
* @param color		Color of line
* @param layer		Layer number (Layer_1, Layer_2)
* @param buffer		Buffer number (Buffer_1, Buffer_2)
*
*/
void DMA2D_Draw_Y_Line(uint16_t xpos, uint16_t ypos, uint16_t length, uint16_t width, uint32_t color, Layer_e layer, Buffer_e buffer);

/**
* @brief Draw a line
*
* @param xpos 		x-position
* @param ypos 		y-position
* @param length 	Length of line
* @param width		Width of line
* @param color		Color of line
* @param layer		Layer number (Layer_1, Layer_2)
* @param buffer		Buffer number (Buffer_1, Buffer_2)
*
*/
void DMA2D_Draw_Line(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1, uint32_t color, Layer_e layer, Buffer_e buffer);

/**
* @brief Draw a rectangle
*
* @param xpos 		x-position
* @param ypos 		y-position
* @param length 	Length of rectangle in pixels
* @param height		Height of rectangle in pixels
* @param color		Color of rectangle
* @param layer		Layer number (Layer_1, Layer_2)
* @param buffer		Buffer number (Buffer_1, Buffer_2)
*
*/
void DMA2D_Draw_Rectangle(uint16_t xpos, uint16_t ypos, uint16_t length, uint16_t height, uint32_t color, uint8_t thickness, Layer_e layer, Buffer_e buffer);

/**
* @brief Draw text
*
* @param string		Text to be written
* @param xpos 		x-position
* @param ypos 		y-position
* @param color		Font color
* @param font		Font
* @param layer		Layer number (Layer_1, Layer_2) Foreground
* @param buffer		Buffer number (Buffer_1, Buffer_2) Foreground
* @param bglayer	Layer number (Layer_1, Layer_2) Background
* @param bgbuffer	Buffer number (Buffer_1, Buffer_2) Background
* @param waitForVsync		If true, wait for vertical sync in ltdc->CDSR register
*/
void DMA2D_write_string(char* string, uint16_t xpos, uint16_t ypos, uint32_t color, tFont* font, Layer_e layer, Buffer_e buffer, bool waitForVsync);


/**
* @brief Set the alpha value of the given layer
*
* @param layer Layer number (Layer_1, Layer_2)
* @param alpha Alpha value (0-255)
*/
void LTDC_setLayerAlpha(Layer_e layer, uint8_t alpha);

/**
* @brief Select the active Display buffer
*
* @param buffer The buffer to activate (Buffer_1, Buffer_2)
*/
void LTDC_Switch_Buffer(Buffer_e buffer);


#endif /* __D_DISPLAY_DMA2D_HAL_H_ */
