/***********************************************************
 *
 * d_Display_DMA2D_HAL.c
 * SYSTART GmbH
 * MIGA, 26.10.2021
 *
 ***********************************************************/

/***********************************************************
 *
 * @description: This file implements the Buffer and Layer
 * handling used in the main and application code.
 * One of the two buffers can be used to draw a frame in the
 * background while displaying the other buffer.
 * The two buffers are located in the SDRAM of the board.
 * Every buffer has two layers, which are merged together,
 * using the alpha values of the layer.
 * Layer 2 is the front layer and overlays Layer 1.
 * e.g. If layer 2 uses max alpha channel value, layer 1 is
 * invisible.
 * CubeMX          | our code
 * LTDC_LAYER_1(0) = Layer 1
 * LTDC_LAYER_2(1) = Layer 2
 *
 *      ---------------             ---------------
 *     /   Buffer 1   /            /   Buffer 2   /
 *    /              /------      /              /------
 *   /   Layer 2    /      / <-> /   Layer 2    /      /
 *   ---------------      /      ---------------      /
 *        /   Layer 1    /            /   Layer 1    /
 *       ----------------            ----------------
 *
 * 1 Buffer and 2 Layers active
 * Address    | Name
 * 0xC0000000		Buffer1-Layer1
 * 0xC0200000		Buffer1-Layer2
 * 2 Buffers and 1 Layer active
 * Address    | Name
 * 0xC0000000		Buffer1-Layer1
 * 0xC0200000		Buffer2-Layer1
 * 2 Buffer and 2 Layers active
 * Address    | Name
 * 0xC0000000   Buffer1-Layer1
 * 0xC0200000   Buffer2-Layer1
 * 0xC0400000   Buffer1-Layer2
 * 0xC0600000   Buffer2-Layer2
 *
 * In 1 layer mode, the forground layer (Layer_2) is disabled
 * and only the background layer (Layer_1) is available to
 * draw onto.
 *
 ***********************************************************/

/******************************
 * Includes
 *****************************/
#include <stdlib.h>
#include "driver/d_Display_DMA2D_HAL.h"
#include "../images/ebssystart_logo.h"
#include "../images/2in1_displaylogo.h"

extern DMA2D_HandleTypeDef hdma2d;
extern LTDC_HandleTypeDef hltdc;

#ifndef _swap_int16_t
	#define _swap_int16_t(a, b) { int16_t swap = a; a = b; b = swap; }
#endif


/******************************
 * Function definitions
 *****************************/
Buffer_e Display_Startup_Sequence(){

		Layer_e workinglayer = (LCD_COUNT_LAYER == 2) ? Layer_2 : Layer_1;

	  LTDC_Switch_Buffer(Buffer_1);
	  if(LCD_BPP == 2){
	  DMA2D_Fill_Color(0xFFFF, Layer_1, Buffer_1, false);
	  DMA2D_Fill_Color(0x7FFF, workinglayer, Buffer_1, false);
	  }
	  else if(LCD_BPP == 4){
	  	DMA2D_Fill_Color(0xFFFFFFFF, Layer_1, Buffer_1, false);
			DMA2D_Fill_Color(0x00FFFFFF, workinglayer, Buffer_1, false);
	  }
	  LTDC_setLayerAlpha(workinglayer, 255);
	  DMA2D_Draw_Image((HDP-320)/2, (VDP-240)/2, 320, 240, 255, DMA2D_REPLACE_ALPHA, (uint32_t) &image_data_ebssystart_logo, CM_ARGB8888, workinglayer, Buffer_1, false);
	  if(LCD_BPP == 2){
	  DMA2D_Fill_Color(0xFFFF, Layer_1, Buffer_2, false);
	  DMA2D_Fill_Color(0x7FFF, workinglayer, Buffer_2, false);
	  }
	  else if(LCD_BPP == 4){
			DMA2D_Fill_Color(0xFFFFFFFF, Layer_1, Buffer_2, false);
			DMA2D_Fill_Color(0x00FFFFFF, workinglayer, Buffer_2, false);
		}
	  DMA2D_Draw_Image((HDP-300)/2, (VDP-123)/2, 300, 123, 255, DMA2D_REPLACE_ALPHA, (uint32_t) &image_data_2in1_display_logo, CM_ARGB8888, workinglayer, Buffer_2, false);
	  HAL_Delay(2000);
	  LTDC_Switch_Buffer(Buffer_2);
	  LTDC_setLayerAlpha(workinglayer, 255);
	  HAL_Delay(2000);
	  return Buffer_2;

}


void DMA2D_DrawPixel (uint16_t xpos, uint16_t ypos, uint32_t color, Layer_e layer, Buffer_e buffer){
	if(xpos >= LCD_X_PIXEL || ypos >= LCD_Y_PIXEL) return;

	if(LCD_BPP == 4 || LCD_BPP == 3) 	*(__IO uint32_t *) (LCD_START_ADDR + LCD_COUNT_BUFFER * layer * LCD_BUFFER_SIZE + buffer * LCD_BUFFER_SIZE + ypos * LCD_X_PIXEL * LCD_BPP + xpos * LCD_BPP) = color;
	else if(LCD_BPP == 2) 				*(__IO uint16_t *) (LCD_START_ADDR + LCD_COUNT_BUFFER * layer * LCD_BUFFER_SIZE + buffer * LCD_BUFFER_SIZE + ypos * LCD_X_PIXEL * LCD_BPP + xpos * LCD_BPP) = (uint16_t) color;
}

void DMA2D_Fill_Color(uint32_t color, Layer_e layer, Buffer_e buffer, bool waitForVsync){
	uint32_t clr;
	hdma2d.Instance = DMA2D;
  hdma2d.Init.Mode = DMA2D_R2M;
  if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB565){
    hdma2d.Init.ColorMode = DMA2D_RGB565;
  	clr = 	0xff << 24 								|
  					((color >> 10) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x3f) * 0xff / 0x3f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB888){
    hdma2d.Init.ColorMode = DMA2D_RGB888;
    clr = (0xff*color+127)/255;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB1555){
  	hdma2d.Init.ColorMode = DMA2D_ARGB1555;
  	clr = 	((color >>15) * 255) << 24 									|
  					((color >> 9) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x1f) * 0xff / 0x1f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB8888){
    hdma2d.Init.ColorMode = DMA2D_ARGB8888;
    clr=color;
  }

  hdma2d.Init.OutputOffset = 0;

  HAL_DMA2D_DeInit(&hdma2d);
  HAL_DMA2D_Init(&hdma2d);
  if(waitForVsync){
  while(hltdc.Instance->CDSR & LTDC_CDSR_HSYNCS);
  }
  HAL_DMA2D_Start(&hdma2d, clr, LCD_START_ADDR + LCD_COUNT_BUFFER*layer*LCD_BUFFER_SIZE+buffer*LCD_BUFFER_SIZE, LCD_X_PIXEL, LCD_Y_PIXEL);
  while(hdma2d.Instance->CR & DMA2D_CR_START);
}

void DMA2D_Draw_FilledRectangle(uint32_t color, uint16_t xpos, uint16_t ypos, uint16_t width, uint16_t height, Layer_e layer, Buffer_e buffer, bool waitForVsync){
	uint32_t clr;
	hdma2d.Instance = DMA2D;
  hdma2d.Init.Mode = DMA2D_R2M;
  if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB565){
    hdma2d.Init.ColorMode = DMA2D_RGB565;
  	clr = 	0xff << 24 								|
  					((color >> 10) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x3f) * 0xff / 0x3f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB888){
    hdma2d.Init.ColorMode = DMA2D_RGB888;
    clr = (0xff*color+127)/255;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB1555){
  	hdma2d.Init.ColorMode = DMA2D_ARGB1555;
  	clr = 	((color >>15) * 255) << 24 									|
  					((color >> 9) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x1f) * 0xff / 0x1f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB8888){
    hdma2d.Init.ColorMode = DMA2D_ARGB8888;
    clr=color;
  }

  hdma2d.Init.OutputOffset = LCD_X_PIXEL-width;
  HAL_DMA2D_DeInit(&hdma2d);
  HAL_DMA2D_Init(&hdma2d);
  if(waitForVsync){
  while(hltdc.Instance->CDSR & LTDC_CDSR_HSYNCS);
  }
  HAL_DMA2D_Start(&hdma2d, clr, LCD_START_ADDR+ LCD_COUNT_BUFFER*layer*LCD_BUFFER_SIZE+buffer*LCD_BUFFER_SIZE +xpos*LCD_BPP + ypos*LCD_X_PIXEL*LCD_BPP, width, height);
  HAL_DMA2D_PollForTransfer(&hdma2d, 10);
}



void DMA2D_Draw_Image(uint16_t xpos, uint16_t ypos, uint16_t xsize, uint16_t ysize, uint32_t alpha, uint32_t alpha_mode, uint32_t addr_image, uint32_t source_format, Layer_e layer, Buffer_e buffer, bool waitForVsync){
  if(xpos >= LCD_X_PIXEL || ypos >= LCD_Y_PIXEL) return;

  /* Blend existing Layer contents with new image */
  hdma2d.Instance = DMA2D;
  hdma2d.Init.Mode = DMA2D_M2M_BLEND;
  if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB565){
    hdma2d.Init.ColorMode = DMA2D_OUTPUT_RGB565;
    hdma2d.LayerCfg[0].InputColorMode = DMA2D_INPUT_RGB565;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB888){
    hdma2d.Init.ColorMode = DMA2D_OUTPUT_RGB888;
    hdma2d.LayerCfg[0].InputColorMode = DMA2D_INPUT_RGB888;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB1555){
  	hdma2d.Init.ColorMode = DMA2D_ARGB1555;
  	hdma2d.LayerCfg[0].InputColorMode = DMA2D_INPUT_ARGB1555;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB8888){
    hdma2d.Init.ColorMode = DMA2D_OUTPUT_ARGB8888;
    hdma2d.LayerCfg[0].InputColorMode = DMA2D_INPUT_ARGB8888;
  }
  hdma2d.Init.OutputOffset = LCD_X_PIXEL-xsize;

  hdma2d.LayerCfg[1].InputOffset = 0;
  hdma2d.LayerCfg[1].InputColorMode = source_format;
  hdma2d.LayerCfg[1].AlphaMode = alpha_mode;
  hdma2d.LayerCfg[1].InputAlpha = alpha;

  hdma2d.LayerCfg[0].InputOffset = LCD_X_PIXEL-xsize;
  hdma2d.LayerCfg[0].AlphaMode = DMA2D_NO_MODIF_ALPHA;
  hdma2d.LayerCfg[0].InputAlpha = 255;

  HAL_DMA2D_DeInit(&hdma2d);
  HAL_DMA2D_Init(&hdma2d);
  HAL_DMA2D_ConfigLayer(&hdma2d, 0);
  HAL_DMA2D_ConfigLayer(&hdma2d, 1);
  if(waitForVsync){
  while(hltdc.Instance->CDSR & LTDC_CDSR_HSYNCS);
  }
  HAL_DMA2D_BlendingStart(&hdma2d, addr_image, LCD_START_ADDR + LCD_COUNT_BUFFER * layer * LCD_BUFFER_SIZE + buffer * LCD_BUFFER_SIZE + ypos * LCD_X_PIXEL * LCD_BPP + xpos * LCD_BPP ,LCD_START_ADDR + LCD_COUNT_BUFFER * layer * LCD_BUFFER_SIZE + buffer * LCD_BUFFER_SIZE + ypos * LCD_X_PIXEL * LCD_BPP + xpos * LCD_BPP, xsize, ysize);
  while(hdma2d.Instance->CR & DMA2D_CR_START);
}

void DMA2D_Draw_X_Line(uint16_t xpos, uint16_t ypos, uint16_t length, uint16_t width, uint32_t color, Layer_e layer, Buffer_e buffer){
	if(xpos >= LCD_X_PIXEL || ypos >= LCD_Y_PIXEL) return;
	uint32_t clr;
	hdma2d.Instance = DMA2D;
  hdma2d.Init.Mode = DMA2D_R2M;
  if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB565){
    hdma2d.Init.ColorMode = DMA2D_RGB565;
  	clr = 	0xff << 24 								|
  					((color >> 10) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x3f) * 0xff / 0x3f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB888){
    hdma2d.Init.ColorMode = DMA2D_RGB888;
    clr = (0xff*color+127)/255;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB1555){
  	hdma2d.Init.ColorMode = DMA2D_ARGB1555;
  	clr = 	((color >>15) * 255) << 24 									|
  					((color >> 9) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x1f) * 0xff / 0x1f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB8888){
    hdma2d.Init.ColorMode = DMA2D_ARGB8888;
    clr=color;
  }

  hdma2d.Init.OutputOffset = LCD_X_PIXEL-length;

  /* Correct for display size */
	if((xpos+length)>LCD_X_PIXEL){
		length =  LCD_X_PIXEL - xpos;
		hdma2d.Init.OutputOffset = xpos;
	}

  if(ypos < (width-1)/2){
  	width = ypos + width/2;
  	ypos = 0;
  }
	else if(LCD_Y_PIXEL-ypos < (width-1)/2){
		width = (LCD_Y_PIXEL - ypos + width / 2);
	}


  HAL_DMA2D_DeInit(&hdma2d);
  HAL_DMA2D_Init(&hdma2d);
  HAL_DMA2D_Start(&hdma2d, clr, LCD_START_ADDR + LCD_COUNT_BUFFER*layer*LCD_BUFFER_SIZE+buffer*LCD_BUFFER_SIZE +xpos*LCD_BPP + ypos*LCD_X_PIXEL*LCD_BPP, length, width);
  HAL_DMA2D_PollForTransfer(&hdma2d, 10);
}

void DMA2D_Draw_Y_Line(uint16_t xpos, uint16_t ypos, uint16_t length, uint16_t width, uint32_t color, Layer_e layer, Buffer_e buffer){
	if(xpos >= LCD_X_PIXEL || ypos >= LCD_Y_PIXEL) return;
	uint32_t clr;
	hdma2d.Instance = DMA2D;
  hdma2d.Init.Mode = DMA2D_R2M;
  if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB565){
    hdma2d.Init.ColorMode = DMA2D_RGB565;
  	clr = 	0xff << 24 								|
  					((color >> 10) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x3f) * 0xff / 0x3f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB888){
    hdma2d.Init.ColorMode = DMA2D_RGB888;
    clr = (0xff*color+127)/255;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB1555){
  	hdma2d.Init.ColorMode = DMA2D_ARGB1555;
  	clr = 	((color >>15) * 255) << 24 									|
  					((color >> 9) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x1f) * 0xff / 0x1f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB8888){
    hdma2d.Init.ColorMode = DMA2D_ARGB8888;
    clr=color;
  }


  /* Correct for display size */
  if(xpos < (width-1)/2){
		width = xpos+(width-1)/2;
		xpos = 0;
	}
  else if(xpos + width > LCD_X_PIXEL){
		width = LCD_X_PIXEL - xpos;
	}
  else{
  	xpos = xpos - (width-1)/2;
  }

  hdma2d.Init.OutputOffset = LCD_X_PIXEL-width;

  if((ypos+length)>LCD_Y_PIXEL){
		length =  LCD_Y_PIXEL - ypos;
	}

  HAL_DMA2D_DeInit(&hdma2d);
  HAL_DMA2D_Init(&hdma2d);
  HAL_DMA2D_Start(&hdma2d, clr, LCD_START_ADDR + LCD_COUNT_BUFFER*layer*LCD_BUFFER_SIZE+buffer*LCD_BUFFER_SIZE +xpos*LCD_BPP + ypos*LCD_X_PIXEL*LCD_BPP, width, length);
  HAL_DMA2D_PollForTransfer(&hdma2d, 10);
}

void DMA2D_Draw_Line(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1, uint32_t color, Layer_e layer, Buffer_e buffer)
{
  int16_t steep = abs(y1 - y0) > abs(x1 - x0);

  if (steep)
  {
    _swap_int16_t(x0, y0);
    _swap_int16_t(x1, y1);
  }
  if (x0 > x1)
  {
	_swap_int16_t(x0, x1);
	_swap_int16_t(y0, y1);
  }

  int16_t dx, dy;
  dx = x1 - x0;
  dy = abs(y1 - y0);
  int16_t err = dx / 2;
  int16_t ystep;

  if (y0 < y1)
	  ystep = 1;
  else
	  ystep = -1;

  for (; x0 <= x1; x0++)
  {
	  if (steep)
	  {
		  DMA2D_DrawPixel (y0, x0-1, color, layer, buffer);
		  DMA2D_DrawPixel (y0, x0,   color, layer, buffer);
		  DMA2D_DrawPixel (y0, x0+1, color, layer, buffer);
		  DMA2D_DrawPixel (y0-1, x0,   color, layer, buffer);
		  DMA2D_DrawPixel (y0+1, x0,   color, layer, buffer);
	  }
	   else
	   {
		   DMA2D_DrawPixel (x0, y0-1, color, layer, buffer);
		   DMA2D_DrawPixel (x0, y0,   color, layer, buffer);
		   DMA2D_DrawPixel (x0, y0+1, color, layer, buffer);
		   DMA2D_DrawPixel (x0-1, y0,   color, layer, buffer);
		   DMA2D_DrawPixel (x0+1, y0,   color, layer, buffer);
	   }
	   err -= dy;
	   if (err < 0)
	   {
	      y0 += ystep;
	      err += dx;
	    }
  }
}

void DMA2D_Draw_Rectangle(uint16_t xpos, uint16_t ypos, uint16_t length, uint16_t height, uint32_t color, uint8_t thickness, Layer_e layer, Buffer_e buffer){
	uint8_t i = thickness/2;
	while(i-- != 0)
	{
		DMA2D_Draw_X_Line(xpos , ypos + i, length+1,2, color, layer, buffer); // top line
		DMA2D_Draw_X_Line(xpos , ypos+height - i, length+1,2, color, layer, buffer); // bottom line

		DMA2D_Draw_Y_Line(xpos + i, ypos ,height,2, color, layer, buffer); // left line
		DMA2D_Draw_Y_Line(xpos+length -i, ypos , height,2, color, layer, buffer); // right line
	}
}

void DMA2D_write_string(char* string, uint16_t xpos, uint16_t ypos, uint32_t color, tFont* font, Layer_e layer, Buffer_e buffer, bool waitForVsync){
  uint16_t x_offset = 0, x_add;
  uint32_t clr;

  /* Start Blending the layer */
  hdma2d.Instance = DMA2D;
	hdma2d.Init.Mode = DMA2D_M2M_BLEND;

  if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB565){
    hdma2d.Init.ColorMode = DMA2D_RGB565;
  	clr = 	0xff << 24 								|
  					((color >> 10) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x3f) * 0xff / 0x3f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
		hdma2d.LayerCfg[0].InputColorMode = DMA2D_INPUT_RGB565;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB888){
    hdma2d.Init.ColorMode = DMA2D_RGB888;
    clr = (0xff*color+127)/255;
		hdma2d.LayerCfg[0].InputColorMode = DMA2D_INPUT_RGB888;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB1555){
  	hdma2d.Init.ColorMode = DMA2D_ARGB1555;
  	clr = 	((color >>15) * 255) << 24 									|
  					((color >> 9) & 0x1f) * 0xff / 0x1f  << 16	|
  					((color >> 4) & 0x1f) * 0xff / 0x1f  << 8		|
  					(color & 0x1f) * 0xff / 0x1f;
  	hdma2d.LayerCfg[0].InputColorMode = DMA2D_INPUT_ARGB1555;
  }
  else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB8888){
    hdma2d.Init.ColorMode = DMA2D_ARGB8888;
    clr=color;
		hdma2d.LayerCfg[0].InputColorMode = DMA2D_INPUT_ARGB8888;
  }

	hdma2d.LayerCfg[1].InputOffset = 0;
	hdma2d.LayerCfg[1].InputColorMode = DMA2D_INPUT_A8;
	hdma2d.LayerCfg[1].AlphaMode = DMA2D_NO_MODIF_ALPHA;
	hdma2d.LayerCfg[1].InputAlpha = clr;

	hdma2d.LayerCfg[0].AlphaMode = DMA2D_NO_MODIF_ALPHA;


  while(*string != '\0'){
		if(xpos >= LCD_X_PIXEL || ypos >= LCD_Y_PIXEL) return;
		hdma2d.Init.OutputOffset = LCD_X_PIXEL - font->chars[*string - 0x20].image->width;
		hdma2d.LayerCfg[0].InputOffset = LCD_X_PIXEL - font->chars[*string - 0x20].image->width;
		HAL_DMA2D_DeInit(&hdma2d);
		HAL_DMA2D_Init(&hdma2d);
		HAL_DMA2D_ConfigLayer(&hdma2d, 0);
		HAL_DMA2D_ConfigLayer(&hdma2d, 1);

	  if(waitForVsync){
	  while(hltdc.Instance->CDSR & LTDC_CDSR_HSYNCS);
	  }
		HAL_DMA2D_BlendingStart(
				&hdma2d,
				(uint32_t)font->chars[*string-0x20].image->data,
				LCD_START_ADDR + LCD_COUNT_BUFFER * layer * LCD_BUFFER_SIZE + buffer * LCD_BUFFER_SIZE + ypos * LCD_X_PIXEL * LCD_BPP + (xpos + x_offset) * LCD_BPP,
				LCD_START_ADDR + LCD_COUNT_BUFFER * layer * LCD_BUFFER_SIZE + buffer * LCD_BUFFER_SIZE + ypos * LCD_X_PIXEL * LCD_BPP + (xpos + x_offset) * LCD_BPP,
				font->chars[*string - 0x20].image->width,
				font->chars[*string - 0x20].image->height);

	  while(hdma2d.Instance->CR & DMA2D_CR_START){}

		x_add = font->chars[*string- 0x20].image->width;
    if(x_add < 0) return;
    x_offset += x_add;
    string++;
  }
}

void LTDC_setLayerAlpha(Layer_e layer, uint8_t alpha){
	HAL_LTDC_SetAlpha(&hltdc, alpha, layer);
}

/*
 * Address    | Name
 * 0xC0000000   Buffer1-Layer1
 * 0xC0200000   Buffer2-Layer1
 * 0xC0400000   Buffer1-Layer2
 * 0xC0600000   Buffer2-Layer2
 *
 */

void LTDC_Switch_Buffer(Buffer_e buffer){

  LTDC_LayerCfgTypeDef pLayerCfg = {0};

  if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB565){
  	pLayerCfg.PixelFormat = LTDC_PIXEL_FORMAT_RGB565;
	}
	else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_RGB888){
		pLayerCfg.PixelFormat = LTDC_PIXEL_FORMAT_RGB888;
	}
	else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB1555){
		pLayerCfg.PixelFormat = LTDC_PIXEL_FORMAT_ARGB1555;
	}
	else if(PIXEL_FORMAT == LTDC_PIXEL_FORMAT_ARGB8888){
		pLayerCfg.PixelFormat = LTDC_PIXEL_FORMAT_ARGB8888;
	}

  pLayerCfg.WindowX0 = 0;
  pLayerCfg.WindowX1 = LCD_X_PIXEL;
  pLayerCfg.WindowY0 = 0;
  pLayerCfg.WindowY1 = LCD_Y_PIXEL;
  pLayerCfg.Alpha = 255;
  pLayerCfg.Alpha0 = 255;
  pLayerCfg.BlendingFactor1 = LTDC_BLENDING_FACTOR1_PAxCA;
  pLayerCfg.BlendingFactor2 = LTDC_BLENDING_FACTOR2_PAxCA;
  pLayerCfg.FBStartAdress = LCD_START_ADDR + buffer * LCD_BUFFER_SIZE;
  pLayerCfg.ImageWidth = LCD_X_PIXEL;
  pLayerCfg.ImageHeight = LCD_Y_PIXEL;
  pLayerCfg.Backcolor.Blue = 255;
  pLayerCfg.Backcolor.Green = 255;
  pLayerCfg.Backcolor.Red = 255;

  while(hltdc.Instance->CDSR & LTDC_CDSR_HSYNCS);

  HAL_LTDC_ConfigLayer(&hltdc, &pLayerCfg, LTDC_LAYER_1);

  if(LCD_COUNT_LAYER == 2){
  pLayerCfg.Alpha = 255;
  pLayerCfg.Alpha0 = 255;
  pLayerCfg.FBStartAdress = LCD_START_ADDR + buffer * LCD_BUFFER_SIZE + 2 * LCD_BUFFER_SIZE;

  HAL_LTDC_ConfigLayer(&hltdc, &pLayerCfg, LTDC_LAYER_2);
  }
}

