#include "bsec.h"
#include <SPI.h>
#include <WiFi.h>
#include "ESPAsyncWebServer.h"
#include "SPIFFS.h"

Bsec iaqSensor;
SPIClass hspi;

char input;

union float_bytes_union{
  byte bytes[4];
  float floatVal;
};

float_bytes_union float_bytes;

float temperature;  // deg Celsius
float humidity;     //%
float pressure;     //hPa
float approx_altitude;  //meters
float gas_resistance;   //kOhm
float breath_voc; //ppm
float iaq_value;
float co2_equivalent; //ppm

// Replace with your own network credentials
const char* apSSID = "Weather Station";
const char* apPASSWORD = "";  // only needed for key protected network
IPAddress apIP(192, 168, 0, 1);

AsyncWebServer server(80);
//AsyncWebSocket ws("/websocket");

float calcAltitude(float atmospheric, float seaLevel)
{
  // Equation taken from BMP180 datasheet (page 16):
  //  http://www.adafruit.com/datasheets/BST-BMP180-DS000-09.pdf

  // Note that using the equation from wikipedia can give bad results
  // at high altitude. See this thread for more information:
  //  http://forums.adafruit.com/viewtopic.php?f=22&t=58064

  return 44330.0 * (1.0 - pow(atmospheric / seaLevel, 0.1903));
}

void setup() {
  Serial.begin(115200);
  hspi.begin();
  iaqSensor.begin(15, hspi);

  bsec_virtual_sensor_t sensorList[10] = {
    BSEC_OUTPUT_RAW_TEMPERATURE,  // iaqSensor.rawTemperature
    BSEC_OUTPUT_RAW_PRESSURE, // iaqSensor.pressure
    BSEC_OUTPUT_RAW_HUMIDITY, // iaqSensor.rawHumidity
    BSEC_OUTPUT_RAW_GAS,  // iaqSensor.gasResistance
    BSEC_OUTPUT_IAQ,  // iaqSensor.iaq
    BSEC_OUTPUT_STATIC_IAQ, // iaqSensor.staticIaq
    BSEC_OUTPUT_CO2_EQUIVALENT, // iaqSensor.co2Equivalent
    BSEC_OUTPUT_BREATH_VOC_EQUIVALENT,  // iaqSensor.breathVocEquivalent
    BSEC_OUTPUT_SENSOR_HEAT_COMPENSATED_TEMPERATURE,  // iaqSensor.temperature
    BSEC_OUTPUT_SENSOR_HEAT_COMPENSATED_HUMIDITY, // iaqSensor.humidity
  };
  iaqSensor.updateSubscription(sensorList, 10, BSEC_SAMPLE_RATE_LP);

  SPIFFS.begin();
  WiFi.mode(WIFI_AP);
  WiFi.softAPConfig(apIP, apIP, IPAddress(255, 255, 255, 0));
  WiFi.softAP(apSSID); //open network; to set up pre-shared key protected network: WiFi.softAP(appSSID, apPASSWORD)

  //ws.onEvent(onWsEvent);
  //server.addHandler(&ws);

  server.on("/", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(SPIFFS, "/webpage.html", "text/html");
  });
  server.on("/ebssystart_logo", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(SPIFFS, "/ebssystart_logo.jpg", "image/jpg");
  });
  server.on("/temperature", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(200, "text/plain", String(temperature));
  });
  server.on("/humidity", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(200, "text/plain", String(humidity));
  });
  server.on("/pressure", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(200, "text/plain", String(pressure));
  });
  server.on("/altitude", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(200, "text/plain", String(approx_altitude));
  });
  server.on("/gasResistance", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(200, "text/plain", String(gas_resistance));
  });
  
  server.on("/co2equivalent", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(200, "text/plain", String(co2_equivalent));
  });
  server.on("/iaq", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(200, "text/plain", String(iaq_value));
  });
  server.on("/bVOC", HTTP_GET, [](AsyncWebServerRequest *request){
    request->send(200, "text/plain", String(breath_voc));
  });
  //Start server
  server.begin();
}

void loop() {
  if (iaqSensor.run()) { // If new data is available
    return;
  }
  temperature = iaqSensor.temperature;  // deg Celsius
  humidity = iaqSensor.humidity;  //percent
  pressure = iaqSensor.pressure/100.0;  //hPa
  approx_altitude = calcAltitude(pressure, 1013.25);  // meters
  gas_resistance = iaqSensor.gasResistance/1000.0;  //kOhm
  co2_equivalent = iaqSensor.co2Equivalent;  //ppm
  iaq_value = iaqSensor.iaq;
  breath_voc = iaqSensor.breathVocEquivalent;  //ppm

  if(Serial.available()>0){
    input = Serial.read();
    switch(input){
      case 0x00:
        Serial.write(0xAA);
        break;
      case 0x01:
        float_bytes.floatVal = temperature;
        Serial.write(float_bytes.bytes, 4);
        float_bytes.floatVal = humidity;
        Serial.write(float_bytes.bytes, 4);
        float_bytes.floatVal = pressure;
        Serial.write(float_bytes.bytes, 4);
        float_bytes.floatVal = approx_altitude;
        Serial.write(float_bytes.bytes, 4);
        float_bytes.floatVal = gas_resistance;
        Serial.write(float_bytes.bytes, 4);
        break;
      case 0x02:
        float_bytes.floatVal = temperature;
        Serial.write(float_bytes.bytes, 4);
        break;
      case 0x03:
        float_bytes.floatVal = humidity;
        Serial.write(float_bytes.bytes, 4);
        break;
      case 0x04:
        float_bytes.floatVal = pressure;
        Serial.write(float_bytes.bytes, 4);
        break;
      case 0x05:
        float_bytes.floatVal = approx_altitude;
        Serial.write(float_bytes.bytes, 4);
        break;
      case 0x06:
        float_bytes.floatVal = gas_resistance;
        Serial.write(float_bytes.bytes, 4);
        break;
      case 0x07:
        Serial.println(WiFi.softAPIP());
        break;
    }
  }
}
