/***********************************************************
 *
 * d_uart.h
 * SYSTART GmbH
 * FLB, 12.09.2018
 *
 ***********************************************************/
#ifndef DRIVER_D_UART_H_
#define DRIVER_D_UART_H_

/***********************************************************
 * Includes
 ***********************************************************/
#include "stm32f4xx.h"

/***********************************************************
 * Defines
 ***********************************************************/
//--------------------------------------------------------------
// List of all UARTs
// (start with 0)
//--------------------------------------------------------------
typedef enum
{
	COM1 = 0,
  COM3,
  COM6
}UART_NAME_t;

#define  	UART_ANZ   		3
#define 	UART_TIMEOUT 	10000

//--------------------------------------------------------------
// end command for transfer
//--------------------------------------------------------------
typedef enum {
  NONE = 0,  // no end command
  LFCR,      // LineFeed + CarriageReturn (0x0A,0x0D)
  CRLF,      // CarriageReturn + LineFeed (0x0D,0x0A)
  LF,        // only LineFeed (0x0A)
  CR         // only CarriageReturn (0x0D)
}UART_LASTBYTE_t;

//--------------------------------------------------------------
// Reception state
//--------------------------------------------------------------
typedef enum {
  RX_EMPTY = 0,  // nothing received
  RX_READY,      // data is in buffer
  RX_FULL        // RX-Puffer is full
}UART_RXSTATUS_t;


//--------------------------------------------------------------
// Defines for reception
//--------------------------------------------------------------
#define  RX_BUF_SIZE   800   	// Size of RX-buffer in Bytes
#define  RX_FIRST_CHR  0x00		// first allowed character (ASCII-value 0x20)
#define  RX_LAST_CHR   0xFF		// last allowed character (ASCII-value 0x7E)
#define  RX_END_CHR    0x0A  	// end command (ASCII-value) = \n


//Defines for Endianness
typedef enum{
	BIG_ENDIAN_T,
	LITTLE_ENDIAN_T
}Endianness_t;

//--------------------------------------------------------------
// struct for UART_RX
//--------------------------------------------------------------
typedef struct {
  uint8_t rx_buffer[RX_BUF_SIZE]; // RX-buffer (Ring-buffer)
  uint16_t wr_ptr;             		// Write Pointer
  uint16_t rd_ptr;             		// Read Pointer
  UART_RXSTATUS_t status;      		// RX-State
}UART_RX_t;
UART_RX_t UART_RX[UART_ANZ];

/***********************************************************
 * Function declarations
************************************************************/

/**
* @brief Initialize UART, activate interrupt receive and
* prepare ring buffer
*
*/
void uart_init(void);

/**
* @brief Send a byte through UART
*
* @param uart			which UART to use
* @param byte			byte to send
*
*/
void uart_SendByte(UART_NAME_t uart, uint8_t byte);

/**
* @brief Send a string through UART
*
* @param uart			which UART to use
* @param *ptr			pointer to string (null terminated)
* @param end_cmd	which end char/chars to send
*
*/
void uart_SendString(UART_NAME_t uart, uint8_t *ptr, UART_LASTBYTE_t end_cmd);

/**
* @brief Receive a string from the ring buffer end characters are RX_END_CHR
* and the ASCII values '\' and 'n'
*
* @param uart			which UART to use
* @param *ptr			pointer to store the string
*
* @return status of the RX-buffer
*
*/
UART_RXSTATUS_t uart_ReceiveString(UART_NAME_t uart, char *ptr);

/**
* @brief Send a defined array via given UART
*
* @param uart				which UART to use
* @param *data			pointer to array
* @param cnt				byte-count to send
* @param endianness	endianness of the array
*
*/
void uart_SendArray(UART_NAME_t uart, uint8_t *data, uint16_t cnt, Endianness_t endianness);

/**
* @brief Receive an array from the ring buffer, until read and write pointer
* align
*
* @param uart 		which UART to use
* @param *data		pointer to where array is saved
* @return Amount of bytes received
*
*/
uint32_t uart_ReceiveArray(UART_NAME_t uart, uint8_t *data);

/**
* @brief Receive 4 bytes from the ring buffer
*
*	@param uart 			which UART to use
*	@param *data			pointer to where bytes are saved
*	@param endianness	endianness of the 4 bytes
*	@return Amount of bytes received
*
*/
uint8_t uart_Receive4Byte(UART_NAME_t uart, uint8_t *data, Endianness_t endianness);

/**
* @brief Receive 2 bytes from the ring buffer
*
*	@param uart 			which UART to use
*	@param *data			pointer to where bytes are saved
*	@param endianness	endianness of the 2 bytes
*	@return Amount of bytes received
*
*/
uint8_t uart_Receive2Byte(UART_NAME_t uart, uint16_t *data, Endianness_t endianness);

/**
* @brief Receive 1 byte from the ring buffer
*
*	@param uart 			which UART to use
*	@param *data			pointer to where byte is saved
*	@param endianness	endianness of the 1 bytes
*	@return Amount of byte received
*
*/
uint8_t uart_Receive1Byte(UART_NAME_t uart, uint8_t *data);

/**
* @brief Return value of read pointer of given UART
*
* @param uart				which UART to use
* @return Position of the read pointer in the RX-buffer
*
*/
uint16_t get_rd_ptr(UART_NAME_t uart);

/**
* @brief Return value of write pointer of given UART
*
* @param uart				which UART to use
* @return Position of the write pointer in the RX-buffer
*
*/
uint16_t get_wr_ptr(UART_NAME_t uart);

/**
* @brief Set the read pointer of the ring buffer to
* the position of the write pointer of the RX-buffer.
*
* @param uart				which UART to use
*
*/
void reset_ringbuffer(UART_NAME_t uart);

/**
* @brief STM32HAL function callback when the interrupt received the
* given amount of bytes (1). Calls the receive via interrupt
* hal function with one byte and the ringbuffer.wr_ptr as target
*
* @param huart 			which UART to use (HAL UART Handle)
*
*/
void HAL_UART_RxCpltCallback(UART_HandleTypeDef *huart);

#endif /* DRIVER_D_UART_H_ */
